#! /usr/bin/env python

"""
Simple experiment to run fMM on a single problem with different fraction parameters.
"""

import os
import os.path
import platform

from lab.environments import LocalEnvironment, BaselSlurmEnvironment
from lab.experiment import Experiment

from downward.experiment import FastDownwardExperiment
from downward import suites
from downward.reports.wvc import WVCReport


# Suites
SUITE_OPTIMAL = [ # needs: BENCHMARKS_DIR
    'blocks', 'depot', 'driverlog', 'elevators-opt08-strips', 'elevators-opt11-strips',
    'floortile-opt11-strips', 'ged-opt14-strips', 'gripper', 'hiking-opt14-strips', 
    'logistics00', 'miconic', 'nomystery-opt11-strips', 'openstacks-opt08-strips', 
    'openstacks-strips', 'pegsol-08-strips', 'pegsol-opt11-strips', 'psr-small', 'rovers',
    'satellite', 'scanalyzer-08-strips', 'scanalyzer-opt11-strips', 'storage', 
    'termes-opt18-strips','tpp', 'transport-opt08-strips', 'transport-opt11-strips', 
    'transport-opt14-strips', 'trucks-strips', 'zenotravel']
SUITE_PROMISING_SINGLE_GOAL = [ # needs: BENCHMARK_DIR
    'blocks', 'custom', 'depot','driverlog','elevators-opt11-strips',
    'ged-opt14-strips', 'gripper', 'hiking-opt14-strips',
    'maintenance-opt14-adl', 'miconic',   
    'rovers'
]
SUITE_SINGLE_GOAL = [ # needs: BENCHMARK_DIR
    'gripper', 'depot','driverlog','elevators-opt11-strips',
    'ged-opt14-strips', 'hiking-opt14-strips',
    'maintenance-opt14-adl', 
    'miconic', 'custom', 'rovers', 'tile'
]
SUITE_SHORT = [ # needs: BENCHMARK_DIR
    'driverlog:p01.pddl', 'driverlog:p02.pddl'
]


NODE = platform.node()
REMOTE = NODE.endswith(".scicore.unibas.ch") or NODE.endswith(".cluster.bc2.ch")
BENCHMARKS_DIR = os.environ["DOWNWARD_BENCHMARKS"]
BENCHMARK_DIR = os.environ["MASTER_BENCHMARK"]
REPO = os.environ['DOWNWARD_REPO']
REV = 'tip' # Branch here: defaut, tip, specific_revision, tag.
if REMOTE:
    ENV = BaselSlurmEnvironment(email="marvin.buff@unibas.ch")
    SUITE = SUITE_OPTIMAL
    DIR = BENCHMARKS_DIR
else:
    ENV = LocalEnvironment(processes=4)
    SUITE = SUITE_SHORT
    DIR = BENCHMARKS_DIR

ATTRIBUTES = ['wvc','p_star','error','c_star']

# Create a new experiment.
exp = Experiment(environment=ENV)
# Add custom parser.
exp.add_resource('parser', 'parsers/computeWVC-parser.py')
exp.add_resource('computeWVC', '../../computeWVC.py')

DRIVER_OPTIONS = ['--overall-time-limit', '60m', '--overall-memory-limit', '3584M']

algo = 'blind'

algo_front = '../../computeWVC'
name = 'computeWVC'

for task in suites.build_suite(BENCHMARKS_DIR, SUITE):
    run = exp.add_run()
    run.add_resource('domain', task.domain_file, symlink=True)
    run.add_resource('problem', task.problem_file, symlink=True)
    run.add_command(
        name,
        ['{computeWVC}', task.domain+'/{problem}', algo],
        time_limit=90,
        memory_limit=3584)
    # AbsoluteReport needs the following properties:
    # 'domain', 'problem', 'algorithm', 'coverage'.
    run.set_property('domain', task.domain)
    run.set_property('problem', task.problem)
    run.set_property('algorithm', 'marvin')
    # Every run has to have a unique id in the form of a list.
    # The algorithm name is only really needed when there are
    # multiple algorithms.
    run.set_property('id', ['name', task.domain, task.problem])
    # Schedule parser.
    run.add_command('parse', ['{parser}'])

exp.add_step('build', exp.build)

exp.add_step('start', exp.start_runs)

exp.add_fetcher(name='fetch')

exp.add_report(
    WVCReport(attributes=ATTRIBUTES),
    outfile='report.html')

exp.run_steps()
